unit domimpl;
{
/*
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is the Sablopas : Pascal wrapper for Sablotron.
 *
 * More infos at http://www.tekool.com/sablopas - Contact : cespern@free.fr
 *
 * The Initial Developer of the Original Code is Christophe ESPERN
 * Portions created by Christophe ESPERN are Copyright (C) 2001-2002 Christophe
 * ESPERN. All Rights Reserved.
 *
 * Contributor(s):
}

interface

{$IFDEF FPC}
{$MODE DELPHI}
{$ENDIF}

uses Classes,SysUtils,sablot,sdom,shandler,sablotengine;

type

 TDomDocument=class;

 TDomNode=class;

 TDomNodeType=integer;

 DomString=string;

  TDomNodeList = class (TObject)
  private
    FOwner: TDomNode;
  public
    constructor Create(AOwner:TDomNode);
    function Item(index:integer): TDomNode;
    function Length: Integer;
  end;
  
  TDomNode = class (TObject)
  private
    FChildNodes: TDomNodeList;
    FOwnerDoc: TDomDocument;
    FPtr: PDomNode;
    function GetChildNodes: TDomNodeList;
    function GetFirstChild: TDomNode;
    function GetLastChild: TDomNode;
    function GetLocalName: DomString;
    function GetNameSpaceURI: DomString;
    function GetNextSibling: TDomNode;
    function GetNodeName: DomString;
    function GetNodeType: TDomNodeType;
    function GetNodeValue: DomString;
    function GetOwnerDocument: TDomDocument;
    function GetParentNode: TDomNode;
    function GetPrefix: DomString;
    function GetPreviousSibling: TDomNode;
    procedure SetNodeName(const Aname: DomString);
    procedure SetNodeValue(const AValue: DomString);
  public
    constructor Create(AownerDoc:TDomDocument); virtual;
    destructor Destroy; override;
    function appendChild(const newChild: TdomNode): TdomNode;
    function cloneNode(deep:boolean;clone:TDomNode): TDomNode;
    function FindNode(query:string): TDomNode;
    function getAttribute(const name:DomString): DomString;
    function insertBefore(const newChild,refChild: TdomNode): TdomNode;
    function removeChild(const OldChild: TdomNode): TdomNode;
    function replaceChild(const newChild,oldChild: TdomNode): TdomNode;
    procedure setAttribute(const Name:DomString;const value:DomString);
    procedure removeAttribute(const Name:DomString);
    function ToString: string; virtual;
    property ChildNodes: TDomNodeList read GetChildNodes;
    property FirstChild: TDomNode read GetFirstChild;
    property LastChild: TDomNode read GetLastChild;
    property LocalName: DomString read GetLocalName;
    property NameSpaceURI: DomString read GetNameSpaceURI;
    property NextSibling: TDomNode read GetNextSibling;
    property NodeName: DomString read GetNodeName write SetNodeName;
    property NodeType: TDomNodeType read GetNodeType;
    property NodeValue: DomString read GetNodeValue write SetNodeValue;
    property OwnerDocument: TDomDocument read GetOwnerDocument;
    property ParentNode: TDomNode read GetParentNode;
    property Prefix: DomString read GetPrefix;
    property PreviousSibling: TDomNode read GetPreviousSibling;
  end;
  


  TDomElement = class (TDomNode)
  end;
  
  TDomAttribute = class (TDomNode)
  end;
  
  TDomText = class (TDomNode)
  end;
  
  TDomCDataSection = class (TDomNode)
  end;
  
  TDomComment = class (TDomNode)
  end;
  
  TDomProcessingInstruction = class (TDomNode)
  end;
  
 TDomImplementation=class;

 TDomNodeSet=class;

  TDomDocument = class (TDomNode)
  private
    FCreatedNodes: TList;
    FCreatedSets: TList;
    FDomImpl: TDomImplementation;
    function GetNodeSet(Index: Integer): TDomNodeSet;
  protected
    procedure AddDomSet(ADomSet:TDomNodeSet);
    function GetOPNode(n:PDomNode): TDomNode;
    function GetPDomDoc: PDomDocument;
    function GetPSituation: PSituation;
    procedure RemoveDomSet(ADomSet:TDomNodeSet);
  public
    constructor Create(AdomImpl:TDomImplementation);overload;
    destructor Destroy; override;
    function CloneForeignNode(deep:boolean;clone:TDomNode): TDomNode;
    function CreateAttribute(const name:DomString): TDomAttribute;
    function CreateCDATASection(const data:DomString): TDomCDataSection;
    function CreateComment(const data:DomString): TDomComment;
    function CreateElement(const tagName:DomString): TDomElement;
    function CreateProcessingInstruction(const target,data:DomString):
            TDomProcessingInstruction;
    function CreateTextNode(const data:DomString): TDomText;
    procedure DisposeNode(n:TDomNode);
    function ToString: DomString; override;
    function Xql(const query:DomString;currentNode:TDomNode): TDomNodeSet;
    property DomImpl: TDomImplementation read FDomImpl;
    property NodeSet[Index: Integer]: TDomNodeSet read GetNodeSet;
  end;
  
  TDomNodeSet = class (TObject)
  private
    FOwnerDoc: TDomDocument;
    FPList: PDomNodeList;
    FQuery: string;
  public
    constructor Create(AOwnerDoc:TDomDocument); virtual;
    destructor Destroy; override;
    function Item(index:integer): TDomNode;
    function Length: Integer;
    property Query: string read FQuery;
  end;
  
  TDomImplementation = class (TSablotEngine)
  private
    FDocuments: TList;
  protected
    procedure LockDocument(d:TDomDocument);
    procedure RaiseDomError;
  public
    constructor Create(Aowner:TComponent); override;
    destructor Destroy; override;
    procedure AddArgDoc(const argname:string;d:TDomDocument);
    function CreateDocument: TDomDocument;
    procedure FreeDocument(d:TDomDocument);
    function Parse(URI:string): TDomDocument;
    function ParseBuffer(const buffer:DomString): TDomDocument;
    function ParseStylesheet(URI:string): TDomDocument;
    function ParseStylesheetBuffer(const buffer:DomString): TDomDocument;
  end;
  


procedure Register;


implementation

procedure Register;
begin
  RegisterComponents('Tekool', [TDomImplementation]);
end;

procedure FreeNodeCallBack(n:SDomNode);cdecl;
var
 dn:TDomNode;
begin
dn:=TDomNode(SDOM_getNodeInstanceData(n));
if dn<>nil then
  dn.FPtr^:=0;
end;

{ TDomNode }

{
*********************************** TDomNode ***********************************
}
constructor TDomNode.Create(AownerDoc:TDomDocument);
begin
  inherited Create;
  FOwnerDoc:=AOwnerDoc;
  if FOwnerDoc<>self then
   FOwnerDoc.FCreatedNodes.Add(self);
  GetMem(FPtr,SizeOf(PDomNode));
end;

destructor TDomNode.Destroy;
var
  idx: Integer;
begin
   if FChildNodes<>nil then
    FChildNodes.Free;
   if FPtr<>nil then begin
     if FPtr^<>0 then
       if NodeType<>SDOM_DOCUMENT_NODE then begin
        SDOM_setNodeInstanceData(FPtr^,nil);
        SDOM_DisposeNode(FOwnerDoc.GetPSituation^,FPtr^);
       end;
   FreeMem(FPtr,SizeOf(PDomNode));
   end;
   idx:=FOwnerDoc.FCreatedNodes.IndexOf(self);
   if idx<>-1 then
     FOwnerDOc.FCreatedNodes.delete(idx);
  inherited;
end;

function TDomNode.appendChild(const newChild: TdomNode): TdomNode;
begin
  Result:=newchild;
  if  SDom_AppendChild(FOwnerDoc.GetPSituation^,FPtr^,newChild.FPtr^)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
end;

function TDomNode.cloneNode(deep:boolean;clone:TDomNode): TDomNode;
var
  n: PDomNode;
  x: Integer;
begin
  if deep then
   x:=1
  else
   x:=0;
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_cloneNode(FOwnerDoc.GetPSituation^,clone.FPtr^,x,n)<>0 then
     FOwnerDoc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNode.FindNode(query:string): TDomNode;
var
  FList: TDomNodeSet;
begin
  FList:=FOwnerDOc.XQL(query,self);
  if Flist=nil then exit;
  if FList.Length>0 then
   Result:=FList.Item(0)
  else
   Result:=nil;
  FList.Free;
end;

function TDomNode.getAttribute(const name:DomString): DomString;
var
  p: PChar;
begin
  SDOM_GetAttribute(FOwnerDoc.GetPSituation^,FPtr^,name,@p);
  Result:=string(p);
end;

function TDomNode.GetChildNodes: TDomNodeList;
begin
  if FChildNodes=nil then
   FChildNodes:=TDomNodeList.Create(self);
  Result:=FChildNodes;
end;

function TDomNode.GetFirstChild: TDomNode;
var
  n: PDomNode;
begin
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_GetFirstChild(FOwnerDoc.GetPSituation^,FPtr^,n)<>0 then
     FOwnerDOc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNode.GetLastChild: TDomNode;
var
  n: PDomNode;
begin
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_GetLastChild(FOwnerDoc.GetPSituation^,FPtr^,n)<>0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
   except
    if n<>nil then
     Freemem(n);
    raise;
  end;
end;

function TDomNode.GetLocalName: DomString;
var
  p: PChar;
begin
  if SDOM_getNodeLocalName(FOwnerDoc.GetPSituation^,FPtr^,@p)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
  Result:=string(p);
end;

function TDomNode.GetNameSpaceURI: DomString;
var
  p: PChar;
begin
  if SDOM_getNodeNSUri(FOwnerDoc.GetPSituation^,FPtr^,@p)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
  Result:=string(p);
end;

function TDomNode.GetNextSibling: TDomNode;
var
  n: PDomNode;
begin
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_GetNextSibling(FOwnerDoc.GetPSituation^,FPtr^,n)<>0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNode.GetNodeName: DomString;
var
  p: PChar;
begin
  SDOM_getNodeName(FOwnerDoc.GetPSituation^,FPtr^,@p);
  Result:=string(p);
end;

function TDomNode.GetNodeType: TDomNodeType;
var
  p: SDOmNodeType;
begin
  SDOM_getNodeType(FOwnerDoc.GetPSituation^,FPtr^,p);
  Result:=p;
end;

function TDomNode.GetNodeValue: DomString;
var
  p: PChar;
begin
  SDOM_getNodeValue(FOwnerDoc.GetPSituation^,FPtr^,@p);
  Result:=string(p);
end;

function TDomNode.GetOwnerDocument: TDomDocument;
begin
  Result:=FOwnerDoc;
end;

function TDomNode.GetParentNode: TDomNode;
var
  n: PDomNode;
begin
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_GetParentNode(FOwnerDoc.GetPSituation^,FPtr^,n)<>0 then
     FOwnerDOc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNode.GetPrefix: DomString;
var
  p: PChar;
begin
  if SDOM_getNodePrefix(FOwnerDoc.GetPSituation^,FPtr^,@p)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
  Result:=string(p);
end;

function TDomNode.GetPreviousSibling: TDomNode;
var
  n: PDomNode;
begin
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_GetPreviousSibling(FOwnerDoc.GetPSituation^,FPtr^,n)<>0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNode.insertBefore(const newChild,refChild: TdomNode): TdomNode;
begin
  Result:=NewChild;
  if SDom_InsertBefore(FOwnerDoc.GetPSituation^,FPtr^,NewChild.FPtr^,RefChild.FPtr^)<>0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
end;

function TDomNode.removeChild(const OldChild: TdomNode): TdomNode;
begin
  Result:=Oldchild;
  if SDom_RemoveChild(FOwnerDoc.GetPSituation^,FPtr^,OldChild.FPtr^)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
end;

function TDomNode.replaceChild(const newChild,oldChild: TdomNode): TdomNode;
begin
  Result:=OldChild;
  if SDom_ReplaceChild(FOwnerDoc.GetPSituation^,FPtr^,NewChild.FPtr^,OldChild.FPtr^)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
end;

procedure TDomNode.setAttribute(const Name:DomString;const value:DomString);
begin
  if SDOM_SetAttribute(FOwnerDoc.GetPSituation^, FPtr^, name, value) <> 0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
end;

procedure TDomNode.removeAttribute(const Name:DomString);
begin
  if SDOM_removeAttribute(FOwnerDoc.GetPSituation^, FPtr^, Name) <> 0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
end;

procedure TDomNode.SetNodeName(const Aname: DomString);
begin
  if SDOM_setNodeName(FOwnerDoc.GetPSituation^,FPtr^,Aname)<>0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
end;

procedure TDomNode.SetNodeValue(const AValue: DomString);
begin
  if SDOM_setNodeValue(FOwnerDoc.GetPSituation^,FPtr^,Avalue)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError;
end;

function TDomNode.ToString: string;
var
  tmp: PChar;
begin
  SDOM_NodeToString(FOwnerDoc.GetPSituation^,FOwnerDoc.Fptr^,FPtr^,@tmp);
  Result:=String(tmp);
end;


{ TDomDocument }

{
********************************* TDomDocument *********************************
}
constructor TDomDocument.Create(AdomImpl:TDomImplementation);
begin
  FDomImpl:=ADOmimpl;
  FCreatedNodes:=TList.Create;
  FCreatedSets:=TList.Create;
  FDomImpl.FDocuments.add(self);
  inherited Create(self);
end;

destructor TDomDocument.Destroy;
var
  i: Integer;
begin
  while FCreatedNodes.Count>0 do
   TObject(FCreatedNodes[FCreatedNodes.Count-1]).free;
  while FCreatedSets.Count>0 do
   TObject(FCreatedSets[FCreatedSets.Count-1]).free;
  i:=FDomImpl.FDOcuments.IndexOf(self);
  if i<>-1 then
   FDomImpl.FDocuments.Delete(i);
  inherited;
  FCreatedNodes.Free;
  FCreatedSets.Free;
end;

procedure TDomDocument.AddDomSet(ADomSet:TDomNodeSet);
begin
  if FCreatedSets.Indexof(ADomSet)=-1 then
   FCreatedSets.Add(ADomSet);
end;

function TDomDocument.cloneForeignNode(deep:boolean;clone:TDomNode): TDomNode;
var
  n: PDomNode;
  x: Integer;
begin
  if deep then
   x:=1
  else
   x:=0;
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_cloneForeignNode(GetPSituation^,self.Fptr^,clone.FPtr^,x,n)<>0 then
     FOwnerDoc.FDomImpl.RaiseDomError;
   Result:=GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomDocument.createAttribute(const name:DomString): TDomAttribute;
begin
  Result:=TDomAttribute.Create(self);
  SDOM_CreateAttribute(GetPSituation^,GetPDomDoc^,Result.FPtr,name);
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

function TDomDocument.createCDATASection(const data:DomString): 
        TDomCDataSection;
begin
  Result:=TDomCDataSection.Create(self);
  SDOM_CreateCDataSection(GetPSituation^,GetPDomDoc^,Result.FPtr,data);
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

function TDomDocument.createComment(const data:DomString): TDomComment;
begin
  Result:=TDomComment.Create(self);
  SDOM_CreateComment(GetPSituation^,GetPDomDoc^,Result.FPtr,data);
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

function TDomDocument.createElement(const tagName:DomString): TDomElement;
begin
  Result:=TDomElement.Create(self);
  SDOM_CreateElement(GetPSituation^,GetPDomDoc^,Result.FPtr,tagname);
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

function TDomDocument.createProcessingInstruction(const target,data:DomString): 
        TDomProcessingInstruction;
begin
  Result:=TDomProcessingInstruction.Create(self);
  SDOM_CreateProcessingInstruction(GetPSituation^,GetPDomDoc^,Result.FPtr,target,data);
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

function TDomDocument.createTextNode(const data:DomString): TDomText;
begin
  Result:=TDomText.Create(self);
  SDOM_CreateTextNode(GetPSituation^,GetPDomDoc^,Result.FPtr,data);
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

procedure TDomDocument.disposeNode(n:TDomNode);
var
  tmp: TDomNode;
  ns: PDomNode;
begin
  try
   GetMem(ns,SizeOf(PdomNode));
   if n.firstChild<>nil then begin
    ns^:=n.Firstchild.FPtr^;
   while ns^<>0 do begin
     tmp:= TDomNode(SDOM_GetNodeInstanceData(ns^));
     SDOM_GetNextSibling(FOwnerDoc.GetPSituation^,ns^,ns);
     if tmp<>nil then
      FOwnerDoc.DisposeNode(tmp);
    end;
   end;
   FreeMem(ns);
   n.free;
  except
   if ns<>nil then
     Freemem(ns);
   raise;
  end;
end;

function TDomDocument.GetNodeSet(Index: Integer): TDomNodeSet;
begin
  if (index>-1) and (index<FCreatedSets.count) then
   Result:=TDomNodeSet(FCreatedSets[index]);
end;

function TDomDocument.GetOPNode(n:PDomNode): TDomNode;
var
  ntype: SDomNodeType;
begin
  Result:=nil;
  if n^=0 then
   exit;
  try
  Result:= SDOM_GetNodeInstanceData(n^);
  if Result=nil then begin
   SDOM_GetNodeType(GetPSituation^,n^,ntype);
   case ntype of
    SDOM_ELEMENT_NODE:Result:=TDomElement.create(self);
    SDOM_ATTRIBUTE_NODE:Result:=TDomAttribute.Create(self);
    SDOM_TEXT_NODE:Result:=TDomText.Create(self);
    SDOM_CDATA_SECTION_NODE:Result:=TDomCDataSection.Create(self);
    SDOM_PROCESSING_INSTRUCTION_NODE:Result:=TDomProcessingInstruction.Create(self);
    SDOM_COMMENT_NODE:Result:=TDomComment.Create(self);
    else
     Result:=TDomNode.Create(self);
   end;
   Result.Fptr^:=n^;
   SDOM_SetNodeInstanceData(Result.FPtr^,result);
  end
  except
   if Result<>nil then
    Result.Free;
   raise;
  end;
end;

function TDomDocument.GetPDomDoc: PDomDocument;
begin
  Result:=PDomDocument(FPtr);
end;

function TDomDocument.GetPSituation: PSituation;
begin
  Result:=FDomImpl.Situation;
end;

procedure TDomDocument.RemoveDomSet(ADomSet:TDomNodeSet);
begin
  if FCreatedSets.Indexof(ADomSet)<>-1 then
   FCreatedSets.Remove(ADomSet);
end;

function TDomDocument.ToString: DomString;
var
  tmp: PChar;
begin
  SDOM_docToString(GetPSituation^,FPtr^,@tmp);
  Result:=String(tmp);
end;

function TDomDocument.xql(const query:DomString;currentNode:TDomNode): 
        TDomNodeSet;
begin
  try
  Result:=TDomNodeSet.Create(self);
  Result.FQuery:=query;
  if SDOM_XQL(GetPSituation^,PChar(query),currentNode.Fptr^,Result.FPlist)<>0 then
   FDomImpl.RaiseDomError;
  except
  //   FDomImpl.RaiseMessage(MH_LEVEL_ERROR,'Sablopas','XQL query error');
  end;
end;




{ TDomImplementation }

{
****************************** TDomImplementation ******************************
}
constructor TDomImplementation.Create(Aowner:TComponent);
begin
  inherited;
  SDOM_setDisposeCallback(@FreeNodeCallBack);
  FDOcuments:=TList.Create;
end;

destructor TDomImplementation.Destroy;
begin
  while FDocuments.Count>0 do
   TDomDocument(FDocuments[FDocuments.Count-1]).free;
  FDocuments.Free;
  inherited;
end;

procedure TDomImplementation.AddArgDoc(const argname:string;d:TDomDocument);
begin
  if SablotAddArgTree(Situation^,Processor^,Pchar(argName),d.GetPDomDoc^)<>0 then
   RaiseDomError;
end;

function TDomImplementation.CreateDocument: TDomDocument;
begin
  Result:=TDomDocument.Create(self);
  SabCreatePDocument(PDOmDocument(Result.FPtr));
  SDOM_setNodeInstanceData(Result.FPtr^,Result);
end;

procedure TDomImplementation.FreeDocument(d:TDomDocument);
begin
  LockDocument(d);
  d.free;
end;

procedure TDomImplementation.LockDocument(d:TDomDocument);
begin
  SabLockDocument(d.FPtr^)
end;

function TDomImplementation.Parse(URI:string): TDomDocument;
begin
  try
   Result:=TDomDocument.Create(self);
   if SabParse(URI,Result.GetPDomDoc)<>0 then
    raise Exception.create('');
   SDOM_setNodeInstanceData(Result.FPtr^,Result);
  except
   if Result<>nil then
    Result.free;
   Result:=nil;
  end;
end;

function TDomImplementation.ParseBuffer(const buffer:DomString): TDomDocument;
begin
  try
   Result:=TDomDocument.Create(self);
   SabParseBuffer(buffer,Result.GetPDomDoc);
   SDOM_setNodeInstanceData(Result.FPtr^,Result);
  except
   if Result<>nil then
    Result.free;
   Result:=nil;
  end;
end;

function TDomImplementation.ParseStylesheet(URI:string): TDomDocument;
begin
  try
   Result:=TDomDocument.Create(self);
   SabParseStylesheet(URI,Result.GetPDomDoc);
   SDOM_setNodeInstanceData(Result.FPtr^,Result);
  except
   if Result<>nil then
    Result.free;
   Result:=nil;
  end;
end;

function TDomImplementation.ParseStylesheetBuffer(const buffer:DomString): 
        TDomDocument;
begin
  try
   Result:=TDomDocument.Create(self);
   SabParseStylesheetBuffer(buffer,Result.GetPDomDoc);
   SDOM_setNodeInstanceData(Result.FPtr^,Result);
  except
   if Result<>nil then
    Result.free;
   Result:=nil;
  end;
end;

procedure TDomImplementation.RaiseDomError;
begin
  raise Exception.Create(SDOM_GetExceptionMessage(Situation^));
end;


{ TDomNodeSet }

{
********************************* TDomNodeSet **********************************
}
constructor TDomNodeSet.Create(AOwnerDoc:TDomDocument);
begin
  inherited Create;
  FOwnerDoc:=AOwnerDoc;
  FOwnerDoc.AddDomSet(self);
  GetMem(FPlist,SizeOf(PDomNodeList));
end;

destructor TDomNodeSet.Destroy;
begin
  if FPList<>nil then begin
   if FPList^<>0 then
    SDOM_disposeNodeList(FOwnerDoc.GetPSituation^,FPList^);
   FreeMem(FPList);
  end;
  FOwnerDoc.RemoveDomSet(self);
  inherited;
end;

function TDomNodeSet.Item(index:integer): TDomNode;
var
  n: PDomNode;
begin
  try
   GetMem(n,SizeOf(PDomNode));
   if  SDOM_getNodeListItem(FOwnerDoc.GetPSituation^,FPList^,index,n)<>0 then
    FOwnerDoc.FDomImpl.RaiseDomError;
   Result:=FOwnerDoc.GetOPNode(n);
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNodeSet.Length: Integer;
var
  lgth: SDomLength;
begin
  Result:=0;
  if FPList^=0 then
   exit;
  if   SDOM_getNodeListLength(FOwnerDoc.GetPSituation^,FPList^,lgth)<>0 then
   FOwnerDoc.FDomImpl.RaiseDomError
  else
   Result:=lgth;
end;

{ TDomNodeList }

{
********************************* TDomNodeList *********************************
}
constructor TDomNodeList.Create(AOwner:TDomNode);
begin
  FOwner:=AOwner;
end;

function TDomNodeList.Item(index:integer): TDomNode;
var
  n: PDomNode;
begin
   Result:=nil;
  try
   GetMem(n,SizeOf(PDomNode));
   if SDOM_getChildNodeIndex(FOwner.OwnerDocument.GetPSituation^,FOwner.FPtr^,index,n)=0 then
    Result:=FOwner.OwnerDocument.GetOPNode(n)
   else
    FOwner.OwnerDocument.FDomImpl.RaiseDomError;
   Freemem(n);
  except
   if n<>nil then
    Freemem(n);
   raise;
  end;
end;

function TDomNodeList.Length: Integer;
var
  lgth: SDomLength;
begin
  Result:=0;
  if SDOM_getChildNodeCount(FOwner.OwnerDocument.GetPSituation^,FOwner.FPtr^,lgth)<>0 then
   FOwner.OwnerDocument.FDomImpl.RaiseDomError
  else
   Result:=lgth;
end;


end.
